package com.mvw.allchemical;

import java.io.ByteArrayOutputStream;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;

import com.mvw.allchemical.recipe.*;

public abstract class Recipe implements Comparable<Recipe> {
	
	private static ArrayList<Recipe> recipeList = new ArrayList<Recipe>();
	protected static HashMap<Integer, Recipe> idTemplateMap = new HashMap<Integer, Recipe>();
	protected static HashMap<String, Recipe> nameTemplateMap = new HashMap<String, Recipe>();
	protected static HashMap<String, Integer> priorityMap = new HashMap<String, Integer>();

	static {
		Recipe.addTemplate(0, "sizeless", new SizelessRecipe(), 100);
		Recipe.addTemplate(1, "shapeless", new ShapelessRecipe(), 200);
	}

	protected boolean consumes;
	protected Layout layout;
	protected ArrayList<Component> result;
	protected ArrayList<Modifier> modifiers;
	protected ArrayList<String> usageFlag;
	protected String name;
	protected int id;
	
	protected static void addTemplate(int id, String name, Recipe recipe, int priority) {
		recipe.id = id;
		recipe.name = name;
		idTemplateMap.put(id, recipe);
		nameTemplateMap.put(name, recipe);
		priorityMap.put(name, priority);
	}
	
	public static Recipe add(int id, Layout layout, ArrayList<Component> result) {
		return add(id, layout, result, false);
	}
	
	public static Recipe add(int id, Layout layout, ArrayList<Component> result, boolean consumes) {
		return add(id, layout, result, consumes, new ArrayList<Modifier>());
	}
	
	public static Recipe add(int id, Layout layout, ArrayList<Component> result, boolean consumes, ArrayList<Modifier> modifiers) {
		if(!idTemplateMap.containsKey(id)) {
			return null;
		}
		return add(idTemplateMap.get(id), layout, result, consumes, modifiers);
	}

	public static Recipe add(String name, Layout layout, ArrayList<Component> result) {
		return add(name, layout, result, false);
	}
	
	public static Recipe add(String name, Layout layout, ArrayList<Component> result, boolean consumes) {
		return add(name, layout, result, consumes, new ArrayList<Modifier>());
	}
	
	public static Recipe add(String name, Layout layout, ArrayList<Component> result, boolean consumes, ArrayList<Modifier> modifiers) {
		if(!nameTemplateMap.containsKey(name)) {
			return null;
		}
		return add(nameTemplateMap.get(name), layout, result, consumes, modifiers);
	}
	
	private static Recipe add(Recipe templateRecipe, Layout layout, ArrayList<Component> result, boolean consumes, ArrayList<Modifier> modifiers) {
		return add(templateRecipe, layout, result, consumes, modifiers, new ArrayList<String>());
	}
	
	private static Recipe add(Recipe templateRecipe, Layout layout, ArrayList<Component> result, boolean consumes, ArrayList<Modifier> modifiers, ArrayList<String> usageFlag) {
		Recipe recipe = templateRecipe.add(layout, result, consumes, modifiers, usageFlag);
		recipe.id = templateRecipe.id;
		recipe.name = templateRecipe.name;
		recipeList.add(recipe);
		return recipe;
	}
	
	public static ArrayList<Recipe> getAll() {
		return new ArrayList<Recipe>(recipeList);
	}
	
	public static Result getResults(Layout layout) {
		return getResults(layout, null);
	}
	
	public static Result getResults(Layout layout, ArrayList<String> usageFlag) {
		Result result = new Result();
		Layout used = new Layout(layout);
		Collections.sort(recipeList, Collections.reverseOrder());
		boolean[][] unused = new boolean[used.getHeight()][used.getWidth()];
		int[][] consumeQuantity = new int[used.getHeight()][used.getWidth()];
		for(int i = 0; i < used.getHeight() * used.getWidth(); i++) {
			int x = i % used.getWidth();
			int y = i / used.getWidth();
			unused[y][x] = (used.getAt(x, y) != null);
		}
		result.consumedComponents = unused;
		for(int i = 0; i < recipeList.size(); i++) {
			Recipe recipe = recipeList.get(i);
			if(usageFlag != null) {
				boolean hasFlag = usageFlag.isEmpty() && recipe.usageFlag.isEmpty();
				if(!hasFlag) {
					if(recipe.usageFlag.isEmpty() || usageFlag.isEmpty()) {
						continue;
					}
					for(String uFlag : usageFlag) {
						if(recipe.usageFlag.contains(uFlag)) {
							hasFlag = true;
							break;
						}
					}
				}
				if(!hasFlag) {
					continue;
				}
			}
			Result res = recipe.getResult(used, unused);
			if(res.resultComponents.isEmpty()) {
				continue;
			}
			result.resultComponents.addAll(res.resultComponents);
			for(int j = 0; j < used.getHeight() * used.getWidth(); j++) {
				int x = j % used.getWidth();
				int y = j / used.getWidth();
				unused[y][x] = (unused[y][x] && res.consumedComponents[y][x]);
				consumeQuantity[y][x] = Math.max(consumeQuantity[y][x], res.quantity[y][x]);
				if(recipe.consumes) {
					if(!unused[y][x]) {
						int q = Math.max(0, consumeQuantity[y][x] - used.getQuantityAt(x, y));
						consumeQuantity[y][x] = 0;
						if(q > 0) {
							used.setQuantity(q, x, y);
						} else {
							used.removeContent(x, y);
						}
					}
				}
			}
		}
		return result;
	}
	
	public Layout getLayout() {
		return new Layout(layout);
	}
	
	public ArrayList<Component> getRawResult() {
		return new ArrayList<Component>(result);
	}
	
	public Coordinate getSize() {
		return layout.getSize();
	}
	
	public ArrayList<Modifier> getModifiers() {
		return new ArrayList<Modifier>(modifiers);
	}
	
	public byte[] dumpData() {
		ByteArrayOutputStream dataStream = new ByteArrayOutputStream();
		dataStream.write(id & 0xFF);
		dataStream.write((id >> 8) & 0xFF);
		ByteArrayOutputStream subStream = new ByteArrayOutputStream();
		byte[] data = layout.dumpData();
		subStream.write(data, 0, data.length);
		data = Modifier.getComponentsAsBytes(result);
		subStream.write(data, 0, data.length);
		subStream.write((consumes ? 1 : 0));
		data = Modifier.getModifiersAsBytes(modifiers);
		subStream.write(data, 0, data.length);
		data = subStream.toByteArray();
		subStream.reset();
		dataStream.write(data.length & 0xFF);
		dataStream.write((data.length >> 8) & 0xFF);
		dataStream.write((data.length >> 16) & 0xFF);
		dataStream.write((data.length >> 24) & 0xFF);
		dataStream.write(data, 0, data.length);
		data = dataStream.toByteArray();
		dataStream.reset();
		return data;
	}
	
	public static byte[] dumpFullData() {
		Charset encoding = Charset.forName("UTF-8");
		ByteArrayOutputStream dataStream = new ByteArrayOutputStream();
		byte[] str = "RCP".getBytes(encoding);
		dataStream.write(str, 0, str.length);
		dataStream.write(0);
		ByteArrayOutputStream subStream = new ByteArrayOutputStream();
		subStream.write(recipeList.size() & 0xFF);
		subStream.write((recipeList.size() >> 8) & 0xFF);
		subStream.write((recipeList.size() >> 16) & 0xFF);
		subStream.write((recipeList.size() >> 24) & 0xFF);
		byte[] data;
		for(Recipe recipe : recipeList) {
			data = recipe.dumpData();
			subStream.write(data, 0, data.length);
		}
		data = subStream.toByteArray();
		subStream.reset();
		dataStream.write(data.length & 0xFF);
		dataStream.write((data.length >> 8) & 0xFF);
		dataStream.write((data.length >> 16) & 0xFF);
		dataStream.write((data.length >> 24) & 0xFF);
		dataStream.write(data, 0, data.length);
		data = dataStream.toByteArray();
		return data;
	}
	
	@Override
	public int compareTo(Recipe o) {
		if(this == o) {
			return 0;
		}
		if(consumes != o.consumes) {
			return (consumes ? 1 : -1);
		}
		if(priorityMap.get(name) != priorityMap.get(o.name)) {
			return (priorityMap.get(name) > priorityMap.get(o.name) ? 1 : -1);
		}
		Coordinate tSize = getSize();
		Coordinate oSize = o.getSize();
		if(tSize.x * tSize.y != oSize.x * oSize.y) {
			return (tSize.x * tSize.y > oSize.x * oSize.y ? 1 : -1);
		}
		if(tSize.y != oSize.y) {
			return (tSize.y > oSize.y ? 1 : -1);
		}
		if(tSize.x != oSize.x) {
			return (tSize.x > oSize.x ? 1 : -1);
		}
		for(int y = 0; y < tSize.y; y++) {
			for(int x = 0; x < tSize.x; x++) {
				Content tContent = layout.getAt(x, y);
				Content oContent = o.layout.getAt(x, y);
				boolean tNull = tContent == null;
				boolean oNull = oContent == null;
				if(tNull != oNull) {
					return (oNull ? 1 : -1);
				}
				int comp = tContent.compareTo(oContent);
				if(comp != 0) {
					return comp;
				}
			}
		}
		return 0;
	}
	
	protected abstract Recipe add(Layout layout, ArrayList<Component> result, boolean consumes, ArrayList<Modifier> modifiers, ArrayList<String> usageFlag);
	
	public abstract Result getResult(Layout layout, boolean[][] unused);
}
