package com.mvw.allchemical;

import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;

import com.mvw.allchemical.modifier.*;

public abstract class Modifier {
	
	private static HashMap<Integer, Modifier> idMap = new HashMap<Integer, Modifier>();
	private static HashMap<String, Modifier> nameMap = new HashMap<String, Modifier>();
	
	protected String name;
	protected int id;
	
	protected static Modifier createModifier(int id, String name, Modifier modifier) {
		modifier.id = id;
		modifier.name = name;
		idMap.put(id, modifier);
		nameMap.put(name, modifier);
		return modifier;
	}
	
	static {
		createModifier(1, "if", new IfModifier());
		createModifier(2, "not", new NotModifier());
		createModifier(3, "or", new OrModifier());
		createModifier(4, "xor", new XorModifier());
		createModifier(10, "iscomponent", new IsComponentModifier());
		createModifier(11, "equals", new EqualsModifier());
		createModifier(20, "preserve", new PreserveModifier());
	}
	
	protected static byte[] getModifiersAsBytes(ArrayList<Modifier> modifiers) {
		ByteArrayOutputStream dataStream = new ByteArrayOutputStream();
		ByteArrayOutputStream subDataStream = new ByteArrayOutputStream();
		subDataStream.write(modifiers.size() & 0xFF);
		subDataStream.write((modifiers.size() >> 8) & 0xFF);
		for(Modifier modifier : modifiers) {
			byte[] modifierBytes = modifier.getData();
			subDataStream.write(modifierBytes.length & 0xFF);
			subDataStream.write((modifierBytes.length >> 8) & 0xFF);
			subDataStream.write(modifierBytes, 0, modifierBytes.length);
		}
		byte[] sub = subDataStream.toByteArray();
		dataStream.write(sub.length & 0xFF);
		dataStream.write((sub.length >> 8) & 0xFF);
		dataStream.write(sub, 0, sub.length);
		byte[] data = dataStream.toByteArray();
		dataStream.reset();
		return data;
	}
	
	protected static byte[] getMapAsBytes(boolean[][] map) {
		ArrayList<Coordinate> coordinates = new ArrayList<Coordinate>();
		for(int y = 0; y < map.length; y++) {
			for(int x = 0; x < map[y].length; y++) {
				if(map[y][x]) {
					coordinates.add(new Coordinate(x, y));
				}
			}
		}
		return getCoordinatesAsBytes(coordinates);
	}
	
	protected static byte[] getCoordinatesAsBytes(ArrayList<Coordinate> coordinates) {
		ByteArrayOutputStream dataStream = new ByteArrayOutputStream();
		dataStream.write(coordinates.size() & 0xFF);
		dataStream.write((coordinates.size() >> 8) & 0xFF);
		for(int i = 0; i < coordinates.size(); i++) {
			Coordinate coordinate = coordinates.get(i);
			dataStream.write(coordinate.x & 0xFF);
			dataStream.write((coordinate.x >> 8) & 0xFF);
			dataStream.write(coordinate.y & 0xFF);
			dataStream.write((coordinate.y >> 8) & 0xFF);
		}
		byte[] data = dataStream.toByteArray();
		dataStream.reset();
		return data;
	}
	
	protected static byte[] getComponentsAsBytes(ArrayList<Component> components) {
		ByteArrayOutputStream dataStream = new ByteArrayOutputStream();
		dataStream.write(components.size() & 0xFF);
		dataStream.write((components.size() >> 8) & 0xFF);
		for(Component component : components) {
			dataStream.write(component.id & 0xFF);
			dataStream.write((component.id >> 8) & 0xFF);
			dataStream.write((component.id >> 16) & 0xFF);
			dataStream.write((component.id >> 24) & 0xFF);
		}
		byte[] data = dataStream.toByteArray();
		return data;
	}
	
	protected static byte[] getAttributesAsBytes(ArrayList<Attribute> attributes) {
		ByteArrayOutputStream dataStream = new ByteArrayOutputStream();
		dataStream.write(attributes.size() & 0xFF);
		dataStream.write((attributes.size() >> 8) & 0xFF);
		for(Attribute attribute : attributes) {
			dataStream.write(attribute.id & 0xFF);
			dataStream.write((attribute.id >> 8) & 0xFF);
			dataStream.write((attribute.id >> 16) & 0xFF);
			dataStream.write((attribute.id >> 24) & 0xFF);
		}
		byte[] data = dataStream.toByteArray();
		return data;
	}

	protected static byte[] getGroupsAsBytes(ArrayList<Group> groups) {
		ByteArrayOutputStream dataStream = new ByteArrayOutputStream();
		dataStream.write(groups.size() & 0xFF);
		dataStream.write((groups.size() >> 8) & 0xFF);
		for(Group group : groups) {
			dataStream.write(group.id & 0xFF);
			dataStream.write((group.id >> 8) & 0xFF);
			dataStream.write((group.id >> 16) & 0xFF);
			dataStream.write((group.id >> 24) & 0xFF);
		}
		byte[] data = dataStream.toByteArray();
		return data;
	}
	
	protected static ArrayList<Modifier> parseModifiers(String string) {
		ArrayList<Modifier> modifiers = new ArrayList<Modifier>();
		int breakPos = 0;
		int lastBreakPos = 0;
		int indexPos = 0;
		int level = 0;
		Modifier currentModifier;
		while((breakPos = string.indexOf("|", indexPos)) > -1) {
			int tagPos = string.indexOf("[", indexPos);
			int endPos = breakPos;
			if(level > 0) {
				endPos = string.indexOf("]", indexPos);
			}
			if(tagPos < endPos) {
				level++;
				indexPos = tagPos;
			} else {
				if(level > 0) {
					level--;
				} else {
					currentModifier = parseModifier(string.substring(lastBreakPos, breakPos));
					if(currentModifier != null) {
						modifiers.add(currentModifier);
					}
					lastBreakPos = breakPos + 1;
				}
				indexPos = endPos;
			}
			indexPos++;
		}
		currentModifier = parseModifier(string.substring(lastBreakPos));
		if(currentModifier != null) {
			modifiers.add(currentModifier);
		}
		return modifiers;
	}
	
	protected static ArrayList<Modifier> parseModifiers(byte[] bytes) {
		ArrayList<Modifier> modifiers = new ArrayList<Modifier>();
		for(int i = 2; i < bytes.length;) {
			int length = (bytes[i] << 8) | bytes[i + 1];
			byte[] workBytes = Arrays.copyOfRange(bytes, i, i + 2 + length);
			modifiers.add(parseModifier(workBytes));
			i+= 2 + length;
		}
		return modifiers;
	}
	
	protected static Modifier parseModifier(String string) {
		String modifierString = "";
		String data = "";
		int startModifier = string.indexOf("[");
		int endModifier = string.lastIndexOf("]");
		if(startModifier > -1) {
			if(endModifier == -1) {
				return null;
			}
			data = string.substring(startModifier + 1, endModifier);
			modifierString = string.substring(0, startModifier);
		}
		if(!nameMap.containsKey(modifierString)) {
			return null;
		}
		return getInstance(modifierString, data);
	}
	
	protected static Modifier parseModifier(byte[] bytes) {
		if(bytes.length < 2) {
			return null;
		}
		int modifierId = (bytes[2] << 8) | bytes[3];
		byte[] data = new byte[0];
		if(bytes.length > 4) {
			Arrays.copyOfRange(bytes, 4, bytes.length);
		}
		if(!idMap.containsKey(modifierId)) {
			return null;
		}
		return getInstance(modifierId, data);
	}
	
	protected static ArrayList<Coordinate> parseCoordinates(String string) {
		ArrayList<Coordinate> coordinates = new ArrayList<Coordinate>();
		int index = 0;
		while(index < string.length() && (index = string.indexOf("(", index)) > -1) {
			int endIndex = string.indexOf(")", index);
			String subData = string.substring(index + 1, endIndex);
			index = endIndex + 1;
			if(subData.indexOf(",") == -1) {
				continue;
			}
			String[] splitData = subData.split(",");
			int x = Integer.decode(splitData[0]);
			int y = Integer.decode(splitData[1]);
			coordinates.add(new Coordinate(x, y));
		}
		return coordinates;
	}
	
	protected static ArrayList<Coordinate> parseCoordinates(byte[] bytes) {
		ArrayList<Coordinate> coordinates = new ArrayList<Coordinate>();
		int length = (bytes[0] << 8) | bytes[1];
		for(int i = 0; i < length; i++) {
			int x = (bytes[2 + i] << 8) | bytes[3];
			int y = (bytes[4 + i] << 8) | bytes[5];
			coordinates.add(new Coordinate(x, y));
		}
		return coordinates;
	}
	
	protected static ArrayList<Component> parseComponents(String string) {
		ArrayList<Component> components = new ArrayList<Component>();
		String[] splitData = string.split("\\+");
		for(String data : splitData) {
			Component component = Component.get(data);
			if(component != null) {
				components.add(component);
			}
		}
		return components;
	}
	
	protected static ArrayList<Component> parseComponents(byte[] bytes) {
		ArrayList<Component> components = new ArrayList<Component>();
		int length = (bytes[0] << 8) | bytes[1];
		for(int i = 0; i < length; i++) {
			int id = (bytes[2 + i] << 24) | (bytes[3 + i] << 16) | (bytes[4 + i] << 8) | bytes[5 + i];
			components.add(Component.get(id));
		}
		return components;
	}

	protected static ArrayList<Attribute> parseAttribute(String string) {
		ArrayList<Attribute> attributes = new ArrayList<Attribute>();
		String[] splitData = string.split("\\+");
		for(String data : splitData) {
			Attribute attribute = Attribute.get(data);
			if(attribute != null) {
				attributes.add(attribute);
			}
		}
		return attributes;
	}
	
	protected static ArrayList<Attribute> parseAttributes(byte[] bytes) {
		ArrayList<Attribute> attributes = new ArrayList<Attribute>();
		int length = (bytes[0] << 8) | bytes[1];
		for(int i = 0; i < length; i++) {
			int id = (bytes[4 + i] << 8) | bytes[5 + i];
			attributes.add(Attribute.get(id));
		}
		return attributes;
	}

	protected static ArrayList<Group> parseGroup(String string) {
		ArrayList<Group> groups = new ArrayList<Group>();
		String[] splitData = string.split("\\+");
		for(String data : splitData) {
			Group group = Group.get(data);
			if(group != null) {
				groups.add(group);
			}
		}
		return groups;
	}
	
	protected static ArrayList<Group> parseGroup(byte[] bytes) {
		ArrayList<Group> groups = new ArrayList<Group>();
		int length = (bytes[0] << 8) | bytes[1];
		for(int i = 0; i < length; i++) {
			int id = (bytes[4 + i] << 8) | bytes[5 + i];
			groups.add(Group.get(id));
		}
		return groups;
	}
	
	public String getName() {
		return name;
	}
	
	public int getId() {
		return id;
	}
	
	protected static Modifier getInstance(String name, String string) {
		Modifier main = nameMap.get(name);
		Modifier instance = main.getInstance(string);
		instance.id = main.id;
		instance.name = main.name;
		return instance;
	}

	protected static Modifier getInstance(int id, byte[] bytes) {
		Modifier main = idMap.get(id);
		Modifier instance = main.getInstance(bytes);
		instance.id = main.id;
		instance.name = main.name;
		return instance;
	}
	
	protected abstract Modifier getInstance(String string);
	
	protected abstract Modifier getInstance(byte[] bytes);
	
	protected abstract byte[] getData();
}
