package com.mvw.allchemical;

import java.io.ByteArrayOutputStream;
import java.util.ArrayList;

public class Layout {
	private int width;
	private int height;
	private Content[][] content;
	private int[][] quantity;
	private ArrayList<ContentStack> contentList = new ArrayList<ContentStack>();
	private ArrayList<ContentStack> componentList = new ArrayList<ContentStack>();
	private ArrayList<ContentStack> attributeList = new ArrayList<ContentStack>();
	private ArrayList<ContentStack> groupList = new ArrayList<ContentStack>();
	
	public Layout(int width, int height) {
		this.width = width;
		this.height = height;
		
		content = new Content[height][width];
		
		for(int i = 0; i < width * height; i++) {
			int x = i % width;
			int y = i / width;
			content[y][x] = null;
			quantity[y][x] = 0;
		}
	}
	
	public Layout(Content[][] content) {
		width = content[0].length;
		height = content.length;
		this.content = content;
		this.quantity = new int[height][width];
		for(int i = 0; i < width * height; i++) {
			int x = i % width;
			int y = i / width;
			Content c = getAt(x, y);
			if(c != null) {
				quantity[y][x] = 1;
				ContentStack s = new ContentStack(c, 1, new Coordinate(x, y), false);
				contentList.add(s);
				if(c instanceof Component) {
					componentList.add(s);
				}
				if(c instanceof Attribute) {
					attributeList.add(s);
				}
				if(c instanceof Group) {
					groupList.add(s);
				}
			}
		}
	}
	
	public Layout(ArrayList<Content> content) {
		this(content.toArray(new Content[content.size()]));
	}
	
	public Layout(Content...content) {
		width = content.length;
		height = 1;
		this.content = new Content[1][width];
		this.content[0] = content;
		this.quantity = new int[height][width];
		for(int i = 0; i < width; i++) {
			Content c = getAt(i, 0);
			quantity[0][i] = 1;
			ContentStack s = new ContentStack(c, 1, new Coordinate(i, 0), false);
			contentList.add(s);
			if(c instanceof Component) {
				componentList.add(s);
			}
			if(c instanceof Attribute) {
				attributeList.add(s);
			}
			if(c instanceof Group) {
				groupList.add(s);
			}
		}
	}
	
	public Layout(Layout layout) {
		width = layout.width;
		height = layout.height;
		content = new Content[height][width];
		quantity = new int[height][width];
		for(int i = 0; i < width * height; i++) {
			int x = i % width;
			int y = i / width;
			content[y][x] = layout.content[y][x];
			quantity[y][x] = layout.quantity[y][x];
		}
		contentList.addAll(layout.contentList);
		componentList.addAll(layout.componentList);
		attributeList.addAll(layout.attributeList);
		groupList.addAll(layout.groupList);
	}
	
	public void setContent(Content c, int x, int y) {
		if(content[y][x] != null) {
			removeContent(x, y);
		}
		content[y][x] = c;
		quantity[y][x] = 1;
		ContentStack s = new ContentStack(c, 1, new Coordinate(x, y), false);
		contentList.add(s);
		if(c instanceof Component) {
			componentList.add(s);
		}
		if(c instanceof Attribute) {
			attributeList.add(s);
		}
		if(c instanceof Group) {
			groupList.add(s);
		}
	}
	
	public void setQuantity(int q, int x, int y) {
		Content c = getAt(x, y);
		if(c != null) {
			quantity[y][x] = q;
			ContentStack s;
			for(int i = 0; i < contentList.size(); i++) {
				s = contentList.get(i);
				if(s.coordinate.x == x && s.coordinate.y == y) {
					s.quantity = q;
					break;
				}
			}
			ArrayList<ContentStack> sList = new ArrayList<ContentStack>();
			if(c instanceof Component) {
				sList = componentList;
			}
			if(c instanceof Attribute) {
				sList = attributeList;
			}
			if(c instanceof Group) {
				sList = groupList;
			}
			for(int i = 0; i < sList.size(); i++) {
				s = sList.get(i);
				if(s.coordinate.x == x && s.coordinate.y == y) {
					s.quantity = q;
					break;
				}
			}
		}
	}
	
	public void removeContent(int x, int y) {
		Content c = content[y][x];
		if(c == null) {
			return;
		}
		content[y][x] = null;
		ContentStack s;
		for(int i = 0; i < contentList.size(); i++) {
			s = contentList.get(i);
			if(s.coordinate.x == x && s.coordinate.y == y) {
				contentList.remove(i);
				break;
			}
		}
		ArrayList<ContentStack> sList = new ArrayList<ContentStack>();
		if(c instanceof Component) {
			sList = componentList;
		}
		if(c instanceof Attribute) {
			sList = attributeList;
		}
		if(c instanceof Group) {
			sList = groupList;
		}
		for(int i = 0; i < sList.size(); i++) {
			s = sList.get(i);
			if(s.coordinate.x == x && s.coordinate.y == y) {
				sList.remove(i);
				break;
			}
		}
	}
	
	public void removeContent(boolean[][] keepList) {
		if(keepList.length != height || keepList[0].length != width) {
			return;
		}
		for(int y = 0; y < height; y++) {
			for(int x = 0; x < width; x++) {
				if(!keepList[y][x]) {
					removeContent(x, y);
				}
			}
		}
	}
	
	public int getWidth() {
		return width;
	}
	
	public int getHeight() {
		return height;
	}
	
	public Coordinate getSize() {
		return new Coordinate(width, height);
	}
	
	public Content getAt(int x, int y) {
		return content[y][x];
	}
	
	public int getQuantityAt(int x, int y) {
		return quantity[y][x];
	}
	
	public boolean isComponentAt(int x, int y) {
		if(content[y][x] == null) {
			return false;
		}
		return content[y][x] instanceof Component;
	}
	
	public int getContentSize() {
		return contentList.size();
	}
	
	public ArrayList<ContentStack> getContentList() {
		return new ArrayList<ContentStack>(contentList);
	}
	
	public ArrayList<ContentStack> getComponentList() {
		return new ArrayList<ContentStack>(componentList);
	}
	
	public ArrayList<ContentStack> getAttributeList() {
		return new ArrayList<ContentStack>(attributeList);
	}
	
	public ArrayList<ContentStack> getGroupList() {
		return new ArrayList<ContentStack>(groupList);
	}
	
	public Layout flip() {
		Content[][] newContent = new Content[height][width];
		for(int y = 0; y < height; y++) {
			newContent[y] = content[height - y - 1];
		}
		return new Layout(newContent);
	}
	
	public Layout mirror() {
		Content[][] newContent = new Content[height][width];
		for(int y = 0; y < height; y++) {
			for(int x = 0; x < width; x++) {
				newContent[y][x] = content[y][width - x - 1];
			}
		}
		return new Layout(newContent);
	}
	
	@Override
	public int hashCode() {
		int hash = 1;
		hash = hash * 11 + width;
		hash = hash * 13 + height;
		for(int y = 0; y < height; y++) {
			for(int x = 0; x < width; x++) {
				if(content[y][x] == null) {
					hash = hash * 23;
				} else if(content[y][x] instanceof Component) {
					Component component = (Component)content[y][x];
					hash = hash * 31 + component.getId(); 
				} else {
					Attribute attribute = (Attribute)content[y][x];
					hash = hash * 29 + attribute.getId();
				}
				hash = hash * 3 + x;
			}
			hash = hash * 5 + y;
		}
		return hash;
	}
	
	@Override
	public boolean equals(Object object) {
		if(!(object instanceof Layout)) {
			return false;
		}
		Layout layout = (Layout) object;
		if(layout.hashCode() == hashCode()) {
			return true;
		}
		if(height != layout.height || width != layout.width) {
			return false;
		}
		for(int y = 0; y < height; y++) {
			for(int x = 0; x < width; x++) {
				if(content[y][x] != layout.content[y][x]) {
					return false;
				}
			}
		}
		return true;
	}
	
	@Override
	public String toString() {
		String s = "";
		for(int y = 0; y < height; y++) {
			if(y > 0) {
				s+= ",";
			}
			s+= "[";
			for(int x = 0; x < width; x++) {
				if(x > 0) {
					s+= ",";
				}
				Content c = content[y][x]; 
				if(c == null) {
					s+= "-";
				} else {
					if(c instanceof Component) {
						s+= "c:";
					}
					if(c instanceof Attribute) {
						s+= "a:";
					}
					if(c instanceof Group) {
						s+= "g:";
					}
					s+= c.getName();
				}
			}
			s+= "]";
		}
		return s;
	}

	public byte[] dumpData() {
		ByteArrayOutputStream dataStream = new ByteArrayOutputStream();
		dataStream.write(width & 0xFF);
		dataStream.write((width >> 8) & 0xFF);
		dataStream.write(height & 0xFF);
		dataStream.write((height >> 8) & 0xFF);
		for(int x = 0; x < width; x++) {
			for(int y = 0; y < height; y++) {
				int type = 0;
				int id = 0;
				Content c = getAt(x, y);
				int q = getQuantityAt(x, y);
				if(c instanceof Group) {
					type = 1;
				}
				if(c instanceof Attribute) {
					type = 2;
				}
				if(c instanceof Component) {
					type = 3;
				}
				if(c != null) {
					id = c.id;
				}
				dataStream.write(type & 0xFF);
				dataStream.write(id & 0xFF);
				dataStream.write((id >> 8) & 0xFF);
				dataStream.write((id >> 16) & 0xFF);
				dataStream.write((id >> 24) & 0xFF);
				dataStream.write(q & 0xFF);
				dataStream.write((q >> 8) & 0xFF);
				dataStream.write((q >> 16) & 0xFF);
				dataStream.write((q >> 24) & 0xFF);
			}
		}
		byte[] data = dataStream.toByteArray();
		dataStream.reset();
		return data;
	}
	
	public static void loadData(byte[] data) {
		
	}
}
