package com.mvw.allchemical;

import java.io.ByteArrayOutputStream;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;

/**
 * An Attribute defines certain specifics about a component, for example, if it's
 * flammable or if it's cold. All attributes are stored in a map so you could use
 * it later.
 * @author GaryCXJk
 *
 */
public class Attribute extends Content {
	private static HashMap<Integer, Attribute> idMap = new HashMap<Integer, Attribute>();
	private static HashMap<String, Attribute> nameMap = new HashMap<String, Attribute>();
	private static ArrayList<Attribute> attributeList = new ArrayList<Attribute>();
	
	private Attribute(int id, String name, String realName, String description) {
		this.id = id;
		this.name = name;
		this.realName = realName;
		this.description = description;
		idMap.put(id, this);
		nameMap.put(name, this);
		attributeList.add(this);
	}
	
	@Override
	public int order() {
		return 2;
	}
	
	/**
	 * Creates a new Attribute and adds it to the map.
	 * @param id the id associated with the Attribute
	 * @param name the name or label of the Attribute
	 * @return an Attribute with the given data
	 */
	public static Attribute add(int id, String name) {
		return add(id, name, name);
	}
	
	/**
	 * Creates a new Attribute and adds it to the map.
	 * @param id the id associated with the Attribute
	 * @param name the name or label of the Attribute
	 * @param realName the real name of the Attribute
	 * @return an Attribute with the given data
	 */
	public static Attribute add(int id, String name, String realName) {
		return add(id, name, realName, realName);
	}

	/**
	 * Creates a new Attribute and adds it to the map.
	 * @param id the id associated with the Attribute
	 * @param name the name or label of the Attribute
	 * @param realName the real name of the Attribute
	 * @param description a description of the Attribute
	 * @return an Attribute with the given data
	 */
	public static Attribute add(int id, String name, String realName, String description) {
		return new Attribute(id, name, realName, description);
	}
	
	/**
	 * Tries to retrieve an Attribute from the map.
	 * @param id the id associated with the Attribute
	 * @return an Attribute, or null if it does not exist
	 */
	public static Attribute get(int id) {
		return idMap.get(id);
	}
	
	/**
	 * Tries to retrieve an Attribute from the map.
	 * @param name the name or label of the Attribute
	 * @return an Attribute, or null if it does not exist
	 */
	public static Attribute get(String name) {
		return nameMap.get(name);
	}
	
	public static ArrayList<Attribute> getAll() {
		return new ArrayList<Attribute>(attributeList);
	}
	
	public byte[] dumpData() {
		return super.dumpData();
	}
	
	public static byte[] dumpFullData() {
		Charset encoding = Charset.forName("UTF-8");
		ByteArrayOutputStream dataStream = new ByteArrayOutputStream();
		byte[] str = "ATT".getBytes(encoding);
		dataStream.write(str, 0, str.length);
		dataStream.write(0);
		ByteArrayOutputStream subDataStream = new ByteArrayOutputStream();
		subDataStream.write(attributeList.size() & 0xFF);
		subDataStream.write((attributeList.size() >> 8) & 0xFF);
		subDataStream.write((attributeList.size() >> 16) & 0xFF);
		subDataStream.write((attributeList.size() >> 24) & 0xFF);
		for(Attribute att : attributeList) {
			byte[] subData = att.dumpData();
			subDataStream.write(subData.length & 0xFF);
			subDataStream.write((subData.length >> 8) & 0xFF);
			subDataStream.write((subData.length >> 16) & 0xFF);
			subDataStream.write((subData.length >> 24) & 0xFF);
			subDataStream.write(subData, 0, subData.length);
		}
		byte[] allSub = subDataStream.toByteArray();
		subDataStream.reset();
		dataStream.write(allSub.length & 0xFF);
		dataStream.write((allSub.length >> 8) & 0xFF);
		dataStream.write((allSub.length >> 16) & 0xFF);
		dataStream.write((allSub.length >> 24) & 0xFF);
		dataStream.write(allSub, 0, allSub.length);
		byte[] data = dataStream.toByteArray();
		dataStream.reset();
		return data;
	}
}
